/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.reflect;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.StandardSystemProperty;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.io.ByteSource;
import com.google.common.io.CharSource;
import com.google.common.io.Resources;
import com.google.common.reflect.ElementTypesAreNonnullByDefault;
import com.google.common.reflect.Reflection;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckForNull;

@ElementTypesAreNonnullByDefault
public final class ClassPath {
    private static final Logger logger = Logger.getLogger(ClassPath.class.getName());
    private static final Splitter CLASS_PATH_ATTRIBUTE_SEPARATOR = Splitter.on(" ").omitEmptyStrings();
    private static final String CLASS_FILE_NAME_EXTENSION = ".class";
    private final ImmutableSet<ResourceInfo> resources;

    private ClassPath(ImmutableSet<ResourceInfo> immutableSet) {
        this.resources = immutableSet;
    }

    public static ClassPath from(ClassLoader classLoader) throws IOException {
        ImmutableSet<LocationInfo> immutableSet = ClassPath.locationsFrom(classLoader);
        HashSet<File> hashSet = new HashSet<File>();
        for (LocationInfo object : immutableSet) {
            hashSet.add(object.file());
        }
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (LocationInfo locationInfo : immutableSet) {
            builder.addAll(locationInfo.scanResources(hashSet));
        }
        return new ClassPath((ImmutableSet<ResourceInfo>)builder.build());
    }

    public ImmutableSet<ResourceInfo> getResources() {
        return this.resources;
    }

    public ImmutableSet<ClassInfo> getAllClasses() {
        return FluentIterable.from(this.resources).filter(ClassInfo.class).toSet();
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses() {
        return FluentIterable.from(this.resources).filter(ClassInfo.class).filter(ClassInfo::isTopLevel).toSet();
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses(String string) {
        Preconditions.checkNotNull(string);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.getTopLevelClasses()) {
            if (!classInfo.getPackageName().equals(string)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    public ImmutableSet<ClassInfo> getTopLevelClassesRecursive(String string) {
        Preconditions.checkNotNull(string);
        String string2 = string + '.';
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.getTopLevelClasses()) {
            if (!classInfo.getName().startsWith(string2)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    static ImmutableSet<LocationInfo> locationsFrom(ClassLoader classLoader) {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (Map.Entry entry : ClassPath.getClassPathEntries(classLoader).entrySet()) {
            builder.add(new LocationInfo((File)entry.getKey(), (ClassLoader)entry.getValue()));
        }
        return builder.build();
    }

    @VisibleForTesting
    static ImmutableSet<File> getClassPathFromManifest(File file, @CheckForNull Manifest manifest) {
        if (manifest == null) {
            return ImmutableSet.of();
        }
        ImmutableSet.Builder builder = ImmutableSet.builder();
        String string = manifest.getMainAttributes().getValue(Attributes.Name.CLASS_PATH.toString());
        if (string != null) {
            for (String string2 : CLASS_PATH_ATTRIBUTE_SEPARATOR.split(string)) {
                URL uRL;
                try {
                    uRL = ClassPath.getClassPathEntry(file, string2);
                }
                catch (MalformedURLException malformedURLException) {
                    logger.warning("Invalid Class-Path entry: " + string2);
                    continue;
                }
                if (!uRL.getProtocol().equals("file")) continue;
                builder.add(ClassPath.toFile(uRL));
            }
        }
        return builder.build();
    }

    @VisibleForTesting
    static ImmutableMap<File, ClassLoader> getClassPathEntries(ClassLoader classLoader) {
        LinkedHashMap<File, ClassLoader> linkedHashMap = Maps.newLinkedHashMap();
        ClassLoader classLoader2 = classLoader.getParent();
        if (classLoader2 != null) {
            linkedHashMap.putAll(ClassPath.getClassPathEntries(classLoader2));
        }
        for (URL uRL : ClassPath.getClassLoaderUrls(classLoader)) {
            File file;
            if (!uRL.getProtocol().equals("file") || linkedHashMap.containsKey(file = ClassPath.toFile(uRL))) continue;
            linkedHashMap.put(file, classLoader);
        }
        return ImmutableMap.copyOf(linkedHashMap);
    }

    private static ImmutableList<URL> getClassLoaderUrls(ClassLoader classLoader) {
        if (classLoader instanceof URLClassLoader) {
            return ImmutableList.copyOf(((URLClassLoader)classLoader).getURLs());
        }
        if (classLoader.equals(ClassLoader.getSystemClassLoader())) {
            return ClassPath.parseJavaClassPath();
        }
        return ImmutableList.of();
    }

    @VisibleForTesting
    static ImmutableList<URL> parseJavaClassPath() {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (String string : Splitter.on(StandardSystemProperty.PATH_SEPARATOR.value()).split(StandardSystemProperty.JAVA_CLASS_PATH.value())) {
            try {
                try {
                    builder.add(new File(string).toURI().toURL());
                }
                catch (SecurityException securityException) {
                    builder.add(new URL("file", null, new File(string).getAbsolutePath()));
                }
            }
            catch (MalformedURLException malformedURLException) {
                logger.log(Level.WARNING, "malformed classpath entry: " + string, malformedURLException);
            }
        }
        return builder.build();
    }

    @VisibleForTesting
    static URL getClassPathEntry(File file, String string) throws MalformedURLException {
        return new URL(file.toURI().toURL(), string);
    }

    @VisibleForTesting
    static String getClassName(String string) {
        int n = string.length() - CLASS_FILE_NAME_EXTENSION.length();
        return string.substring(0, n).replace('/', '.');
    }

    @VisibleForTesting
    static File toFile(URL uRL) {
        Preconditions.checkArgument(uRL.getProtocol().equals("file"));
        try {
            return new File(uRL.toURI());
        }
        catch (URISyntaxException uRISyntaxException) {
            return new File(uRL.getPath());
        }
    }

    static final class LocationInfo {
        final File home;
        private final ClassLoader classloader;

        LocationInfo(File file, ClassLoader classLoader) {
            this.home = Preconditions.checkNotNull(file);
            this.classloader = Preconditions.checkNotNull(classLoader);
        }

        public final File file() {
            return this.home;
        }

        public ImmutableSet<ResourceInfo> scanResources() throws IOException {
            return this.scanResources(new HashSet<File>());
        }

        public ImmutableSet<ResourceInfo> scanResources(Set<File> set) throws IOException {
            ImmutableSet.Builder<ResourceInfo> builder = ImmutableSet.builder();
            set.add(this.home);
            this.scan(this.home, set, builder);
            return builder.build();
        }

        private void scan(File file, Set<File> set, ImmutableSet.Builder<ResourceInfo> builder) throws IOException {
            try {
                if (!file.exists()) {
                    return;
                }
            }
            catch (SecurityException securityException) {
                logger.warning("Cannot access " + file + ": " + securityException);
                return;
            }
            if (file.isDirectory()) {
                this.scanDirectory(file, builder);
            } else {
                this.scanJar(file, set, builder);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void scanJar(File file, Set<File> set, ImmutableSet.Builder<ResourceInfo> builder) throws IOException {
            JarFile jarFile;
            try {
                jarFile = new JarFile(file);
            }
            catch (IOException iOException) {
                return;
            }
            try {
                for (File file2 : ClassPath.getClassPathFromManifest(file, jarFile.getManifest())) {
                    if (!set.add(file2.getCanonicalFile())) continue;
                    this.scan(file2, set, builder);
                }
                this.scanJarFile(jarFile, builder);
            }
            finally {
                try {
                    jarFile.close();
                }
                catch (IOException iOException) {}
            }
        }

        private void scanJarFile(JarFile jarFile, ImmutableSet.Builder<ResourceInfo> builder) {
            Enumeration<JarEntry> enumeration = jarFile.entries();
            while (enumeration.hasMoreElements()) {
                JarEntry jarEntry = enumeration.nextElement();
                if (jarEntry.isDirectory() || jarEntry.getName().equals("META-INF/MANIFEST.MF")) continue;
                builder.add((Object)ResourceInfo.of(new File(jarFile.getName()), jarEntry.getName(), this.classloader));
            }
        }

        private void scanDirectory(File file, ImmutableSet.Builder<ResourceInfo> builder) throws IOException {
            HashSet<File> hashSet = new HashSet<File>();
            hashSet.add(file.getCanonicalFile());
            this.scanDirectory(file, "", hashSet, builder);
        }

        private void scanDirectory(File file, String string, Set<File> set, ImmutableSet.Builder<ResourceInfo> builder) throws IOException {
            File[] fileArray = file.listFiles();
            if (fileArray == null) {
                logger.warning("Cannot read directory " + file);
                return;
            }
            for (File file2 : fileArray) {
                Object object;
                String string2 = file2.getName();
                if (file2.isDirectory()) {
                    object = file2.getCanonicalFile();
                    if (!set.add((File)object)) continue;
                    this.scanDirectory((File)object, string + string2 + "/", set, builder);
                    set.remove(object);
                    continue;
                }
                object = string + string2;
                if (((String)object).equals("META-INF/MANIFEST.MF")) continue;
                builder.add((Object)ResourceInfo.of(file2, (String)object, this.classloader));
            }
        }

        public boolean equals(@CheckForNull Object object) {
            if (object instanceof LocationInfo) {
                LocationInfo locationInfo = (LocationInfo)object;
                return this.home.equals(locationInfo.home) && this.classloader.equals(locationInfo.classloader);
            }
            return false;
        }

        public int hashCode() {
            return this.home.hashCode();
        }

        public String toString() {
            return this.home.toString();
        }
    }

    public static final class ClassInfo
    extends ResourceInfo {
        private final String className;

        ClassInfo(File file, String string, ClassLoader classLoader) {
            super(file, string, classLoader);
            this.className = ClassPath.getClassName(string);
        }

        public String getPackageName() {
            return Reflection.getPackageName(this.className);
        }

        public String getSimpleName() {
            int n = this.className.lastIndexOf(36);
            if (n != -1) {
                String string = this.className.substring(n + 1);
                return CharMatcher.inRange('0', '9').trimLeadingFrom(string);
            }
            String string = this.getPackageName();
            if (string.isEmpty()) {
                return this.className;
            }
            return this.className.substring(string.length() + 1);
        }

        public String getName() {
            return this.className;
        }

        public boolean isTopLevel() {
            return this.className.indexOf(36) == -1;
        }

        public Class<?> load() {
            try {
                return this.loader.loadClass(this.className);
            }
            catch (ClassNotFoundException classNotFoundException) {
                throw new IllegalStateException(classNotFoundException);
            }
        }

        @Override
        public String toString() {
            return this.className;
        }
    }

    public static class ResourceInfo {
        private final File file;
        private final String resourceName;
        final ClassLoader loader;

        static ResourceInfo of(File file, String string, ClassLoader classLoader) {
            if (string.endsWith(ClassPath.CLASS_FILE_NAME_EXTENSION)) {
                return new ClassInfo(file, string, classLoader);
            }
            return new ResourceInfo(file, string, classLoader);
        }

        ResourceInfo(File file, String string, ClassLoader classLoader) {
            this.file = Preconditions.checkNotNull(file);
            this.resourceName = Preconditions.checkNotNull(string);
            this.loader = Preconditions.checkNotNull(classLoader);
        }

        public final URL url() {
            URL uRL = this.loader.getResource(this.resourceName);
            if (uRL == null) {
                throw new NoSuchElementException(this.resourceName);
            }
            return uRL;
        }

        public final ByteSource asByteSource() {
            return Resources.asByteSource(this.url());
        }

        public final CharSource asCharSource(Charset charset) {
            return Resources.asCharSource(this.url(), charset);
        }

        public final String getResourceName() {
            return this.resourceName;
        }

        final File getFile() {
            return this.file;
        }

        public int hashCode() {
            return this.resourceName.hashCode();
        }

        public boolean equals(@CheckForNull Object object) {
            if (object instanceof ResourceInfo) {
                ResourceInfo resourceInfo = (ResourceInfo)object;
                return this.resourceName.equals(resourceInfo.resourceName) && this.loader == resourceInfo.loader;
            }
            return false;
        }

        public String toString() {
            return this.resourceName;
        }
    }
}

